package gov.va.vamf.scheduling.direct.resource;

import com.agilex.healthcare.veteranappointment.resources.AbstractUserResource;
import gov.va.vamf.scheduling.direct.datalayer.clinic.ClinicDataService;
import gov.va.vamf.scheduling.direct.datalayer.eligibility.ClinicalServiceService;
import gov.va.vamf.scheduling.direct.domain.CdwClinic;
import gov.va.vamf.scheduling.direct.domain.ClinicalService;
import gov.va.vamf.scheduling.direct.domain.TypesOfCare;
import gov.va.vamf.security.v1.VamfUser;
import gov.va.vamf.security.v1.filters.JwtRbacRestricted;
import gov.va.vamf.security.v1.filters.JwtResourceRestricted;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import javax.validation.constraints.NotNull;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.container.ContainerRequestContext;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

@Path("/clinical-services")
@Component
@Scope("request")
public class PatientClinicalServicesResource extends AbstractUserResource {

    @Resource
    ClinicalServiceService clinicalServiceService;

    @Resource
    private ClinicDataService clinicDataService;

    @GET
    @JwtRbacRestricted("Veteran")
    @Path("/types-of-care")
    @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
    public List<TypesOfCare> getTypesOfCare() {
        List<ClinicalService> clinicalServices = clinicalServiceService.fetch();
        List<TypesOfCare> typesOfCareList = new ArrayList<>();

        // copy only id and name and don't return stop codes and other properties
        for (ClinicalService clinicalService: clinicalServices) {
            TypesOfCare typesOfCare = new TypesOfCare();
            typesOfCare.setId(clinicalService.getId());
            typesOfCare.setName(clinicalService.getName());
            typesOfCareList.add(typesOfCare);
        }

        return typesOfCareList;
    }

    @GET
    @Path("/patient/{assigning-authority}/{patient-id}/clinics")
    @Produces({MediaType.APPLICATION_JSON})
    @JwtResourceRestricted
    public Collection<CdwClinic> getClinicsByTypeOfCare(
            @PathParam("assigning-authority") String assigningAuthority,
            @PathParam("patient-id") String patientId,
            @NotNull @QueryParam("clinical-service") final String clinicalServiceId,
            @NotNull @QueryParam("three-digit-code") final String threeDigitCode,
            @NotNull @QueryParam("institution-code") final String institutionCode,
            @Context ContainerRequestContext context
    ) {
        final VamfUser user = getCurrentVamfUser(context);
        final String userIcn = user.getPatient().getIcn();
        return clinicDataService.retrieveDisplayToPatientClinicsByStopCode(
                userIcn,
                threeDigitCode,
                institutionCode,
                clinicalServiceId
        );
    }
}


